/*
 * disdec.c
 *
 * Diagnostic Engine Code disassembler
 */

#include <common/disman.h>
#include <stdlib.h>

struct instr {
    int numbytes;
    char *name;
};

struct instr instrs[0x80] = {
    /* 80 - 87 */
    {1, "halt"},
    {3, "goto"},
    {1, "plus"},
    {1, "not"},
    {1, "rotate"},
    {1, "equals"},
    {1, "nover"},
    {1, "pick"},

    /* 88 - 8f */
    {2, "val1"},
    {1, "fromr"},
    {1, "tor"},
    {1, "rget"},
    {1, "emit"},
    {2, NULL}, /* 8d, sub-dispatch */
    {1, "setv"},
    {1, "call0"},

    /* 90 - 97 */
    {1, "dput"},
    {1, "nput%b"},
    {1, "bput%b"},
    {1, "iput"},
    {1, "dget"},
    {1, "nget%b"},
    {1, "bget%b"},
    {1, "iget"},

    /* 98 - 9f */
    {1, "setmem%b"},
    {1, "checkmem%b"},
    {1, "walkmem%b"},
    {1, "movemem%b"},
    {1, "cmpmem%b"},
    {1, "fromx"},
    {1, "erex"},
    {1, "unrecognized"},

    /* a0 - a7 */
    {1, "nop"},
    {3, "jifz"},
    {1, "minus"},
    {1, "and"},
    {1, "shift"},
    {1, "less"},
    {1, "swap"},
    {1, "dup"},

    /* a8 - af */
    {3, "val2"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "rget1"},
    {1, "unrecognized"},
    {2, NULL}, /* ad, sub-dispatch */
    {1, "getv"},
    {1, "call1"},

    /* b0 - b7 */
    {1, "unrecognized"},
    {1, "nput%h"},
    {1, "bput%h"},
    {1, "getbb"},
    {1, "unrecognized"},
    {1, "nget%h"},
    {1, "bget%h"},
    {1, "unrecognized"},

    /* b8 - bf */
    {1, "setmem%h"},
    {1, "checkmem%h"},
    {1, "walkmem%h"},
    {1, "movemem%h"},
    {1, "cmpmem%h"},
    {1, "tox"},
    {1, "getint"},
    {1, "unrecognized"},

    /* c0 - c7 */
    {1, "return"},
    {3, "jnb"},
    {1, "times"},
    {1, "or"},
    {1, "unrecognized"},
    {1, "uless"},
    {1, "unrecognized"},
    {1, "drop"},

    /* c8 - cf */
    {4, "val3"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "rget2"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "callv"},
    {1, "call2"},

    /* d0 - d7 */
    {1, "unrecognized"},
    {1, "nput%w"},
    {1, "bput%w"},
    {1, "setbb"},
    {1, "unrecognized"},
    {1, "nget%w"},
    {1, "nput%w"},
    {1, "unrecognized"},

    /* d8 - df */
    {1, "setmem%w"},
    {1, "checkmem%w"},
    {1, "walkmem%w"},
    {1, "movemem%w"},
    {1, "cmpmem%w"},
    {1, "getx"},
    {1, "setint"},
    {1, "unrecognized"},

    /* e0 - e7 */
    {1, "wait"},
    {1, "illop"},
    {1, "divmod"},
    {1, "xor"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},

    /* e8 - ef */
    {5, "val4"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "rget3"},
    {1, "qkey"},
    {1, "unrecognized"},
    {1, "exit"},
    {1, "call3"},

    /* f0 - f7 */
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},

    /* f8 - ff */
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "unrecognized"},
    {1, "intfo"},
    {1, "unrecognized"},
};

char *base_8d_names[8] = {
    "movblk",
    "movand",
    "movor",
    "movxor",
    "movnblk",
    "movnand",
    "movnor",
    "movnxor",
};

char *base_ad_names[2] = {
    "qscan",
    "qscanf",
};

int disassemble_instr(u8 *data, long address)
{
    char buffer[80];
    int numbytes;
    
    if (!(data[0] & 0x80)) {
	/* high bit not set, is high byte of absolute function address */
	sprintf(buffer, "$%02x%02x", data[0], data[1]);
	numbytes = 2;
    } else {
	struct instr *instr;

	instr = &instrs[data[0] & 0x7f];
	numbytes = instr->numbytes;

	if (!instr->name) {
	    char *width;
	    int maxinstr;
	    char **instrs;
	    
	    if ((data[1] & 0xf0) == 0x00) {
		width = "%b";
	    } else if ((data[1] & 0xf0) == 0x20) {
		width = "%h";
	    } else if ((data[1] & 0xf0) == 0x40) {
		width = "%w";
	    } else {
		/* FIXME: May want to signal more obvious error */
		width = NULL;
	    }

	    if (data[0] == 0x8d) {
		maxinstr = 7;
		instrs = base_8d_names;
	    } else { /* ad */
		maxinstr = 1;
		instrs = base_ad_names;
	    }

	    if ((data[1] & 0x0f) > maxinstr) {
		sprintf(buffer, "unrecognized");
	    } else {
		sprintf(buffer, "%s%s", instrs[data[1] & 0x0f], width);
	    }
	} else if (numbytes == 1) {
	    sprintf(buffer, "%s", instr->name);
	} else if (numbytes == 2) {
	    if (data[0] == 0x88) {
		sprintf(buffer, "%s #x%02x", instr->name, data[1]);
	    } else {
		/* sub-dispatch */
		sprintf(buffer, "sub-dispatch %02x", data[1]);
	    }
	} else if (numbytes == 3) {
	    if (data[0] == 0xa8) {
		sprintf(buffer, "%s #x%02x%02x", instr->name,
			data[1], data[2]);
	    } else {
		sprintf(buffer, "%s $%02x%02x", instr->name,
			data[1], data[2]);
	    }
	} else if (numbytes == 4) {
	    sprintf(buffer, "%s #x%02x%02x%02x", instr->name,
		    data[1], data[2], data[3]);
	} else {
	    sprintf(buffer, "%s #x%02x%02x%02x%02x", instr->name,
		    data[1], data[2], data[3], data[4]);
	}
    }

    if (numbytes == 1) {
	printf("%08lx: %02x              ", address, data[0]);
    } else if (numbytes == 2) {
	printf("%08lx: %02x %02x           ", address, data[0], data[1]);
    } else if (numbytes == 3) {
	printf("%08lx: %02x %02x %02x        ", address,
	       data[0], data[1], data[2]);
    } else if (numbytes == 4) {
	printf("%08lx: %02x %02x %02x %02x     ", address,
	       data[0], data[1], data[2], data[3]);
    } else if (numbytes == 5) {
	printf("%08lx: %02x %02x %02x %02x %02x  ", address,
	       data[0], data[1], data[2], data[3], data[4]);
    }
    
    printf("%s\n", buffer);
    
    return numbytes;
}

/* EOF */
